/************************************************************************
 * NAME:	grt
 *
 * DESCR:	Group Reservation Table implementation.  This implements
 *		the reading and writing of the GRT.  It is normally
 *		cached and writen when necessary.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/

#include "hdos.h"

/************************************************************************
 * NAME:	hdos_grt_read() & hdos_grt_write()
 *
 * DESCR:	Reads and writes the GRT.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- requires that the label has already been read
 ************************************************************************/
int
hdos_grt_read(struct hdosfs *hdosfs)
{
    return(hdos_getlsect(hdosfs,hdosfs->labelbuf.grtsec,hdosfs->GRT));
}

int
hdos_grt_write(struct hdosfs *hdosfs)
{
    return(hdos_putlsect(hdosfs,hdosfs->labelbuf.grtsec,hdosfs->GRT));
}

/************************************************************************
 * NAME:	hdos_grt_build_freelist()
 *
 * DESCR:	Build the freelist into the GRT.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- requires that the GRT and RGT have been read in, and
 *		  that the directory has been read.
 *		- the RGT is consulted to figure out the free list
 *		- The first 2 groups are skipped (they're reserved) when
 *		  looking for free groups.
 ************************************************************************/
void
hdos_grt_build_freelist(struct hdosfs *hdosfs)
{
    int			i;
    int			g = 0;

    for (i=2; i < HDOS_GROUPS(hdosfs); i++ ) {
	if (hdosfs->RGT[i] == 0) {
	    hdosfs->GRT[g] = i;
	    g = i;
	}
    }
    hdosfs->GRT[g] = 0;
}

/************************************************************************
 * NAME:	hdos_grt_free_groups()
 *
 * DESCR:	Returns a count of the free groups.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
hdos_grt_free_groups(struct hdosfs *hdosfs)
{
    int	g = 0;
    int	count = 0;

    do {
	count++;
	g = hdosfs->GRT[g];
    } while (g != 0);

    return(count-1);	/* group 0 really isn't on the free list	*/
}

/************************************************************************
 * NAME:	hdos_grt_alloc_group()
 *
 * DESCR:	Allocates and returns a group off of the free list.
 *
 * ARGS:	
 *
 * RETURNS:	the allocated group, or 0 if no more space
 *
 * NOTES:	- the group is marked as in use in the RGT too.
 ************************************************************************/
int
hdos_grt_alloc_group(struct hdosfs *hdosfs)
{
    int	rg;

    rg = hdosfs->GRT[0];		/* get the top of the free list	*/

    hdosfs->GRT[0] = hdosfs->GRT[rg];	/* reset top to the next free	*/

    hdosfs->GRT[rg] = 0;		/* mark this as the "last"	*/
    hdosfs->RGT[rg] += 1;		/*   and update the RGT		*/

    return(rg);
}

/************************************************************************
 * NAME:	hdos_grt_free_chain()
 *
 * DESCR:	Frees the chain of file groups.  All of the freed blocks
 *		are placed on the free list in the GRT.  Further, the
 *		RGT is updated.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- the blocks are place in the free list in the same order
 *		  that they were.
 *		- the old mechanism is below, I didn't like the fact that
 *		  it put the blocks on the free list in "reverse" order.
 *		  Somehow I felt better putting them on as a group.
 *
 *				int	next;
 *
 *    				for ( ; g != 0; g = next) {
 *				    next = hdosfs->GRT[g];
 *				    hdosfs->RGT[g] -= 1;
 *				    hdosfs->GRT[g] = hdosfs->GRT[0];
 *				    hdosfs->GRT[0] = g;
 *				}

 ************************************************************************/
void
hdos_grt_free_chain(struct hdosfs *hdosfs, int first)
{
    int	last, i;

    /* first find the end of the chain, while updating the RGT	*/

    for (i = first; i != 0; last = i, i = hdosfs->GRT[i])
	hdosfs->RGT[i] -= 1;

    /* at this point, last points to the last block in the chain	*/

    hdosfs->GRT[last] = hdosfs->GRT[0];
    hdosfs->GRT[0] = first;
}

/************************************************************************
 * NAME:	hdos_grt_link()
 *
 * DESCR:	Links one grt to another (for building chains).
 *
 * ARGS:	
 *
 * RETURNS:	nothing...
 *
 * NOTES:	
 ************************************************************************/
void
hdos_grt_link(struct hdosfs *hdosfs, int existing_grp, int new_grp)
{
    hdosfs->GRT[existing_grp] = new_grp;
}

